using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

namespace ImageSaver
{
    public partial class MainWindow : Form
    {
        // for hiding/restoring the pic preview box nicely:
        private int pictPreviewWidth;
        private int pictPreviewHieght;
        private int pictHideMainWindowWidth;

        // for hiding/restoring the log/text window nicely:
        private int debugLogHeight;
        private int debugLogXpos;
        private int debugLogYpos;
        private Point resize_change_txbx;
        private Point resize_change_lstbox;

        private bool convertingImages;

        public MainWindow()
        {
            InitializeComponent();
            this.chkHideLogTxt.Tag = true;  // this control's tag mean it's either "minimized"/false or "expanded"/true

            this.chkMinWindow.Tag = true;
            this.chkRemoveSelected.Tag = true;
            this.chkSelectAllFileNames.Tag = true;
            this.chkClearList.Tag = true;

            this.convertingImages = false;


            // Event handlers:
            this.rdoBMP.CheckedChanged += new EventHandler(this.rdoFileType_CheckedChanged);
            this.rdoGIF.CheckedChanged += new EventHandler(this.rdoFileType_CheckedChanged);
            this.rdoICO.CheckedChanged += new EventHandler(this.rdoFileType_CheckedChanged);
            this.rdoJPG.CheckedChanged += new EventHandler(this.rdoFileType_CheckedChanged);
            this.rdoPNG.CheckedChanged += new EventHandler(this.rdoFileType_CheckedChanged);
            this.rdoTIFF.CheckedChanged += new EventHandler(this.rdoFileType_CheckedChanged);

            this.lstFiles.DragEnter += new DragEventHandler(lstFiles_DragEnter);
            this.lstFiles.DragDrop += new DragEventHandler(lstFiles_DragDrop);

            // User response:
            this.lstFiles.KeyDown += new KeyEventHandler(lstFiles_KeyDown);
            this.lstFiles.MouseUp += new MouseEventHandler(lstFiles_MouseClick);

            this.toolTipFileNameBox.SetToolTip(this.lstFiles, "Right Click on a file's name to open Windows Explorer to it");
            this.toolTipPNGcrush.SetToolTip(this.chkPNGCrush, "Runs the resulting .png file through an optimaztion process, which results in a smaller file size.\nAlso results in a longer conversion process.");
            this.toolTipDebugTextButton.SetToolTip(this.chkHideLogTxt, "Expand or shrink the log window.");
            this.toolTipWindowMin.SetToolTip(this.chkMinWindow, "Compact the window and hide the picture preview");
           
            this.ResizeEnd += new EventHandler(MainWindow_ResizeEnd);
            this.SizeChanged += new EventHandler(MainWindow_SizeChanged);

            // Variables/Fields for this form:
            this.pictPreviewHieght = this.pictPreview.Height;
            this.pictPreviewWidth = this.pictPreview.Width;
            this.pictHideMainWindowWidth = this.Width - this.pictPreview.Width;

            this.debugLogHeight = this.txbxDebugLog.Height;
            this.debugLogXpos = this.txbxDebugLog.Location.X;
            this.debugLogYpos = this.txbxDebugLog.Location.Y;

            this.resize_change_txbx = new Point(this.Width, this.Height);
            this.resize_change_lstbox = new Point(this.Width, this.Height);

            rdoFileType_CheckedChanged(new object(), new System.EventArgs());


        }



        void lstFiles_MouseClick(object sender, MouseEventArgs e)
        {

            if ((e.Button == MouseButtons.Left) || (e.Button == MouseButtons.Middle))
            {
                if (this.lstFiles.Items.Count == 0)
                {
                    this.txbxDebugLog.Text = "Press the  Get Files >>  button to add files into the list\r\n\tor\r\nDrag files from Windows Explorer onto the list";
                    this.btn_GetFiles.ForeColor = System.Drawing.Color.AliceBlue;
                }
            }
            else if (e.Button == MouseButtons.Right)
            {

                // On right clicks, the item that was right clicked will have Windows Explorer open to that file's folder
                int index = this.lstFiles.IndexFromPoint(e.Location);
                if (index != System.Windows.Forms.ListBox.NoMatches)
                {
                    string filename = (string)this.lstFiles.Items[index];

                    // Open up Windows Explorer to this folder!
                    System.Diagnostics.Process launchExplorerProc = new System.Diagnostics.Process();

                    launchExplorerProc.StartInfo.FileName = "\"explorer.exe\"";

                    // construct the folder path of the file name that was clicked on
                    //int slash_index = filename.LastIndexOf('\\');
                    //string file_folder = filename.Substring(0, filename.Length - (filename.Length - slash_index));
                    //launchExplorerProc.StartInfo.Arguments = "\"" + file_folder + "\"";

                    launchExplorerProc.StartInfo.Arguments = "/select,\"" + filename + "\"";

                    //launchExplorerProc.StartInfo.CreateNoWindow = true;
                    launchExplorerProc.StartInfo.ErrorDialog = true;

                    try
                    {
                        launchExplorerProc.Start();
                        this.txbxDebugLog.Text = "Windows Explorer Launched:\r\n\t" + launchExplorerProc.StartInfo.FileName + "\r\n\t" + launchExplorerProc.StartInfo.Arguments;

                    }
                    catch (System.Exception exp)
                    {
                        this.txbxDebugLog.Text = exp.Message + " : Error opening Windows Explorer to file\r\n\t" + launchExplorerProc.StartInfo.FileName + "\r\n\t" + launchExplorerProc.StartInfo.Arguments;
                    }
                }
            }
        }

        void lstFiles_KeyDown(object sender, KeyEventArgs e)
        {
            
            switch (e.KeyCode)
            {
                case Keys.Delete:
                    // removes all selected filesnames from the listbox
                    while (this.lstFiles.SelectedIndices.Count > 0)
                    {
                        this.lstFiles.Items.RemoveAt(this.lstFiles.SelectedIndices[0]);
                    }

                    this.tlstpFilesSize.Text = "None Selected";
                    break;
                case Keys.Escape:
                    // deselects everything in the listbox
                    while (this.lstFiles.SelectedIndices.Count > 0)
                    {
                        this.lstFiles.SetSelected(this.lstFiles.SelectedIndices[0], false);
                    }
                    break;
            }
        }


        void lstFiles_DragEnter(object sender, DragEventArgs e)
        {
           
            if (e.Data.GetDataPresent(DataFormats.FileDrop) == true)
            {
                // we accept only drag/drop from files, so accept this
                e.Effect = DragDropEffects.Copy;    // the initiating control can see this flag and do stuff according to its value
            }
            else
            {
                e.Effect = DragDropEffects.None;    // the initiating control can see this flag and do stuff according to its value
            }
        }


        void lstFiles_DragDrop(object sender, DragEventArgs e)
        {
            // have .Net convert the "FileDrop" drag data into file-name-paths. This results in an array of strings!
            string[] filenames = (string[])e.Data.GetData("FileDrop", true);

            // now add all the file names to the list box of file names!
            this.lstFiles.BeginUpdate();
            foreach (string filepath in filenames)
            {
                if (this.lstFiles.Items.Contains(filepath) == false)
                    this.lstFiles.Items.Add(filepath);
            }
            this.lstFiles.EndUpdate();
        }

        void MainWindow_SizeChanged(object sender, EventArgs e)
        {
            if (this.WindowState == FormWindowState.Maximized)
            {
                this.chkMinWindow.Visible = false;                
            }
            else if (this.WindowState == FormWindowState.Normal)
            {
                this.chkMinWindow.Visible = true;
            }
        }

        // save the size of the picture preview so we can toggle it on/off and have it toggle on nicely
        void MainWindow_ResizeEnd(object sender, EventArgs e)
        {
            if (this.pictPreview.Visible == true)
            {
                // only save the size of the Picture box if it's visible.
                this.pictPreviewHieght = this.pictPreview.Height;
                this.pictPreviewWidth = this.pictPreview.Width;
            }

        }

        // Retrieves the codec info that Microsoft Windows GDI+ has built into it...
        private System.Drawing.Imaging.ImageCodecInfo GetImageCodecInfo(string MIME_type)
        {
            System.Drawing.Imaging.ImageCodecInfo[] codecInfos = System.Drawing.Imaging.ImageCodecInfo.GetImageEncoders();
            foreach (System.Drawing.Imaging.ImageCodecInfo codec in codecInfos)
            {
                if (codec.MimeType.Equals(MIME_type, StringComparison.OrdinalIgnoreCase) == true)
                    return codec;
            }
            return null;
        }

        private void rdoFileType_CheckedChanged(object sender, System.EventArgs e)
        {
            this.grpImgFormats.ForeColor = System.Drawing.SystemColors.ControlText;

            this.txbxDebugLog.Clear();

            string MIME_type = "";

            string status_text = "To ";
            if (this.rdoBMP.Checked == true)
            {
                
                status_text += this.rdoBMP.Text;
                MIME_type = "image/bmp";
                
                this.lblQuality.Visible = false;
                this.numwhlQuality.Visible = false;

                this.chkPNGCrush.Visible = false;
                this.chkPNGCrush.Checked = false;

                this.grpQualitySettings.Visible = true;

                this.lblBitDepth.Visible = true;

                // make 8 and 24 the only options:
                
                this.numwhlBitDepth.Minimum = (decimal)8;
                this.numwhlBitDepth.Maximum = (decimal)24;
                this.numwhlBitDepth.Increment = (decimal)16;
                this.numwhlBitDepth.Value = (decimal)24;
                this.numwhlBitDepth.Visible = true;
                this.numwhlBitDepth.Enabled = true;


                this.txbxDebugLog.Text += "8 bit BMP = saving the file as an 8 bit GIF - then saving that 8 bit GIF into a BMP\r\n\r\n";

            }
            else if (this.rdoGIF.Checked == true)
            {
                status_text += this.rdoGIF.Text;
                MIME_type = "image/gif";

                this.lblQuality.Visible = false;
                this.numwhlQuality.Visible = false;

                this.chkPNGCrush.Visible = false;
                this.chkPNGCrush.Checked = false;

                this.grpQualitySettings.Visible = true;

                // only 8 bit is available:

                this.lblBitDepth.Visible = true;
                this.numwhlBitDepth.Enabled = false;
                this.numwhlBitDepth.Value = (decimal)8;
                this.numwhlBitDepth.Visible = true;
                
                

                this.txbxDebugLog.Text += ".NET .GIF encoder only does 8bit ColorDepth!\r\n\r\n";
            }
            else if (this.rdoICO.Checked == true)
            {
                status_text += this.rdoICO.Text;
                MIME_type = "image/vnd.microsoft.icon";
                this.grpQualitySettings.Visible = false;
                //this.lblQuality.Visible = false;
                //this.numwhlQuality.Visible = false;
                //this.lblBitDepth.Visible = false;
                //this.numwhlBitDepth.Visible = false;
         
            }
            else if (this.rdoJPG.Checked == true)
            {
                status_text += this.rdoJPG.Text;
                MIME_type = "image/jpeg";
                this.grpQualitySettings.Visible = true;

                this.lblQuality.Visible = true;
                this.numwhlQuality.Visible = true;

                this.chkPNGCrush.Visible = false;
                this.chkPNGCrush.Checked = false;

                this.lblBitDepth.Visible = false;
                this.numwhlBitDepth.Visible = false;
                //this.numwhlBitDepth.Enabled = true;
            }
            else if (this.rdoPNG.Checked == true)
            {
                status_text += this.rdoPNG.Text;
                MIME_type = "image/png";
                
                this.lblQuality.Visible = false;
                this.numwhlQuality.Visible = false;

                this.chkPNGCrush.Visible = true;

                // prefer using optipng.exe in the same directory as this program's .exe.
                string optipng_path = System.Environment.CurrentDirectory + "\\optipng.exe";
                if (System.IO.File.Exists(optipng_path) == true)
                {
                    this.chkPNGCrush.Text = "Use optipng";
                    this.chkPNGCrush.Tag = "optipng";
                    this.chkPNGCrush.Enabled = true;
                    this.chkPNGCrush.Checked = true;
                }
                else
                {

                    string pngcrush_path = System.Environment.CurrentDirectory + "\\pngcrush.exe";
                    if (System.Environment.GetCommandLineArgs().Length > 1)
                    {
                        pngcrush_path = (string)(System.Environment.GetCommandLineArgs().GetValue(1));
                    }
                    if (System.IO.File.Exists(pngcrush_path) == true)
                    {
                        this.chkPNGCrush.Text = "Use pngcrush";
                        this.chkPNGCrush.Tag = "pngcrush";
                        this.chkPNGCrush.Enabled = true;
                        this.chkPNGCrush.Checked = true;
                    }
                    else
                    {
                        this.chkPNGCrush.Tag = null;
                        this.chkPNGCrush.Enabled = false;
                        this.chkPNGCrush.Checked = false;
                        this.txbxDebugLog.Text += "Optipng/pngcrush option disabled:\r\n";
                        this.txbxDebugLog.Text += "\tOptipng.exe not found in currenty direcory and\r\n";
                        this.txbxDebugLog.Text += "\tPNGcrush.exe not found in current directory or in 1st command line argument\r\n";
                    }
                }

                

                // only 32 bit is available, but we fake 8 bit, by coverting the original to a gif, then saving that 8bit gif as a 24bit PNG. Reduces file size of PNG.
                // Make "8bit" and 24bit the only options.
                this.lblBitDepth.Visible = true;
                
                this.numwhlBitDepth.Minimum = (decimal)8;
                this.numwhlBitDepth.Maximum = (decimal)32;
                this.numwhlBitDepth.Increment = (decimal)24;
                this.numwhlBitDepth.Value = (decimal)32;
                this.numwhlBitDepth.Visible = true;
                this.numwhlBitDepth.Enabled = true;

                this.grpQualitySettings.Visible = true;

                this.txbxDebugLog.Text = this.txbxDebugLog.Text.Insert(0,"\"8 bit\" PNG is just a 32bit PNG with 256 colors in it = saving the file as an 8bit GIF, then save that 8bit GIF into a PNG\r\n\r\n");
            }
            else if (this.rdoTIFF.Checked == true)
            {
                status_text += this.rdoTIFF.Text;
                MIME_type = "image/tiff";
                
                this.lblQuality.Visible = false;
                this.numwhlQuality.Visible = false;

                this.chkPNGCrush.Visible = false;
                this.chkPNGCrush.Checked = false;

                // Make 24 bit options available, as the .NET TIFF encoder only works with those
                this.lblBitDepth.Visible = true;
                this.numwhlBitDepth.Enabled = false;
                this.numwhlBitDepth.Value = (decimal)24;
                this.numwhlBitDepth.Maximum = (decimal)24;
                this.numwhlBitDepth.Visible = true;
                

                this.grpQualitySettings.Visible = true;
            }

            this.tlstpConversionFormat.Text = status_text;

            // debuging/console-like log:
            

            // required for the BitMap.GetEncoderParameterList()
            Bitmap encoder_bitmap = new Bitmap(1, 1);

            System.Drawing.Imaging.EncoderParameters encoderParams;// = new System.Drawing.Imaging.EncoderParameters(2);
            System.Drawing.Imaging.ImageCodecInfo codecInfo;

            codecInfo = this.GetImageCodecInfo(MIME_type);
            if (codecInfo != null)
            {


                this.txbxDebugLog.Text += "----------" + codecInfo.MimeType + "-------\r\n";

                this.txbxDebugLog.Text += codecInfo.CodecName + "\r\n";

                this.txbxDebugLog.Text += "Version: " + codecInfo.Version.ToString() + "\r\n";

                this.txbxDebugLog.Text += "Filename extensions: " + codecInfo.FilenameExtension + "\r\n";



                try
                {
                    encoderParams = encoder_bitmap.GetEncoderParameterList(codecInfo.Clsid);
                }
                catch (NotImplementedException exp)
                {
                    this.txbxDebugLog.Text += "\r\n" + exp.Message + " : GetEncoderParameterList for mime type \"" + MIME_type + "\" is not implemented yet in the .NET framework";
                    return;
                }
                encoder_bitmap.Dispose();

                this.txbxDebugLog.Text += "---------- Encoder Paramaters ------\r\n";
                foreach (System.Drawing.Imaging.EncoderParameter param in encoderParams.Param)
                {
                    this.txbxDebugLog.Text += param.NumberOfValues.ToString() + " " + param.Type.ToString() + "\r\n";
                }
            }
            else this.txbxDebugLog.Text += "Couldn't find ImageCodecInfo for mime type \"" + MIME_type + "\"\r\n";
           
        }

        private void btn_GetFiles_Click(object sender, EventArgs e)
        {
            this.btn_GetFiles.ForeColor = System.Drawing.SystemColors.ControlText;

            if (this.openFileDialog1.ShowDialog() == System.Windows.Forms.DialogResult.OK)
            {
                this.lstFiles.BeginUpdate();
                foreach (string filename in this.openFileDialog1.FileNames)
                {
                    if(this.lstFiles.Items.Contains(filename) == false)
                        this.lstFiles.Items.Add(filename);
                }
                this.lstFiles.EndUpdate();

                // now select the first file name for the user
                if (this.lstFiles.Items.Count > 0)
                {
                    this.lstFiles.SelectedIndex = 0;

                    this.txbxDebugLog.Text = "<< Choose the image format to convert them to <<";
                    this.grpImgFormats.ForeColor = System.Drawing.Color.AliceBlue;
                }
            }
 
        }

        private void lstFiles_SelectedIndexChanged(object sender, EventArgs e)
        {
            this.pictPreview.ImageLocation = (string)this.lstFiles.SelectedItem;

            // don't update any text if we're done converting and overwriting image files, or else all the good log stuff from the conversion will be replaced!
            if (this.convertingImages == false)
            {
                if (this.lstFiles.SelectedItems.Count == 0)
                {
                    if (this.lstFiles.Items.Count > 0)
                        this.txbxDebugLog.Text = "^^ Select the images from the list you want to convert ^^";
                    else
                    {
                        this.txbxDebugLog.Text = "Press the  Get Files >>  button\r\n\tAnd\r\nSelect files from the list before converting them\r\n";
                        this.btn_GetFiles.ForeColor = System.Drawing.Color.AliceBlue;
                    }

                    this.tlstpFilesSize.Text = "None Selected";
                    return;
                }

                if (this.lstFiles.SelectedItems.Count == 1)
                    this.tlstpFilesSize.Text = "File: 1";
                else
                    this.tlstpFilesSize.Text = "Files: " + this.lstFiles.SelectedItems.Count.ToString();

                this.txbxDebugLog.Text = "<< Choose the image format to convert to";
            }

        }

        private void chkClearList_CheckedChanged(object sender, EventArgs e)
        {
            if ((bool)this.chkClearList.Tag == true)    // prevent having this event come twice in a row
            {
                this.lstFiles.Items.Clear();

                this.chkClearList.Checked = false;

                this.pictPreview.ImageLocation = "";
                this.tlstpFilesSize.Text = "None Selected";
            }
            else
                this.chkClearList.Tag = true;
        }

        private void chkRemoveSelected_CheckedChanged(object sender, EventArgs e)
        {
            if ((bool)this.chkRemoveSelected.Tag == true)
            {
                while (this.lstFiles.SelectedIndices.Count > 0)
                {
                    this.lstFiles.Items.RemoveAt(this.lstFiles.SelectedIndices[0]);
                }

                this.chkRemoveSelected.Checked = false;


                this.tlstpFilesSize.Text = "None Selected";
            }
            else
                this.chkRemoveSelected.Tag = true;
        }

        private void btnConvertImages_Click(object sender, EventArgs e)
        {
            this.grpImgFormats.ForeColor = System.Drawing.SystemColors.ControlText;

            this.txbxDebugLog.Clear();

            this.toolStripProgressBar1.ProgressBar.Value = 25;

            if (this.lstFiles.SelectedIndices.Count == 0)
            {


                if (this.lstFiles.Items.Count == 0)
                {
                    this.txbxDebugLog.Text = "Press the  Get Files >>  button to add files\r\n\tAnd then\r\nSelect files from the list before converting them\r\n";
                    this.btn_GetFiles.ForeColor = System.Drawing.Color.AliceBlue;
                }
                else
                {
                    this.txbxDebugLog.Text = "^^ Select the files you want to convert ^^";
                }

                this.toolStripProgressBar1.ProgressBar.Value = 0;
                return;
            }

            // creating new files from the selected file names in the list box:
            this.Text = "Image Saver (converting)";
            foreach (string filename in this.lstFiles.SelectedItems)
            {
                string open_filename = filename; // make a copy of the iterator/string, so we can modify it
                string save_to_filename;

                // find out what file format to copy into...
                foreach (RadioButton rdo in this.grpImgFormats.Controls)
                {
                    int period_index = open_filename.LastIndexOf('.');
                    int slash_index = open_filename.LastIndexOf('\\');
                    string file_folder = open_filename.Substring(0, open_filename.Length - (open_filename.Length - slash_index));

                    if (rdo.Checked == true)
                    {
                        // construct file name to write to:
                        if ( open_filename.Substring(period_index + 1).ToLowerInvariant() == rdo.Text.ToLowerInvariant())
                        {
                            // they are "converting" the original file to the same type of image format
                            // don't overwrite the file, make a copy with a new name:
                            save_to_filename = open_filename.Substring(0, open_filename.Length - (open_filename.Length - period_index)) + "COPY." + rdo.Text.ToLowerInvariant();
                        }
                        else
                        {
                            // chop off the extension, and add a new one:
                            save_to_filename = open_filename.Substring(0, open_filename.Length - (open_filename.Length - period_index)) + "." + rdo.Text.ToLowerInvariant();
                        }

                        this.toolStripProgressBar1.ProgressBar.Value = 50;

                        // save the image to the new format:
                        //if(rdo.Text == this.rdoBMP.Text) {
                        //    bitmap.Save(save_to_filename, System.Drawing.Imaging.ImageFormat.Bmp);
                        //}
                        //else if(rdo.Text == this.rdoGIF.Text) {
                        //    bitmap.Save(save_to_filename, System.Drawing.Imaging.ImageFormat.Gif);
                        //}
                        //else if(rdo.Text == this.rdoJPG.Text) {
                        //    bitmap.Save(save_to_filename, System.Drawing.Imaging.ImageFormat.Jpeg);
                        //}
                        //else if(rdo.Text == this.rdoPNG.Text) {
                        //    bitmap.Save(save_to_filename, System.Drawing.Imaging.ImageFormat.Png);
                        //}
                        //else if(rdo.Text == this.rdoTIFF.Text) {
                        //    bitmap.Save(save_to_filename, System.Drawing.Imaging.ImageFormat.Tiff);
                        //}
                        //else if(rdo.Text == this.rdoICO.Text) {
                        //    bitmap.Save(save_to_filename, System.Drawing.Imaging.ImageFormat.Icon);
                        //}

                        System.Drawing.Imaging.ImageFormat imageFormat = System.Drawing.Imaging.ImageFormat.Png;
                        string MIME_type = "";

                        if (rdo.Text == this.rdoBMP.Text)
                        {
                            MIME_type = "image/bmp";
                            imageFormat = System.Drawing.Imaging.ImageFormat.Bmp;
                        }
                        else if (rdo.Text == this.rdoGIF.Text)
                        {
                            MIME_type = "image/gif";
                            imageFormat = System.Drawing.Imaging.ImageFormat.Gif;
                        }
                        else if (rdo.Text == this.rdoJPG.Text)
                        {
                            MIME_type = "image/jpeg";
                            imageFormat = System.Drawing.Imaging.ImageFormat.Jpeg;
                        }
                        else if (rdo.Text == this.rdoPNG.Text)
                        {
                            MIME_type = "image/png";
                            imageFormat = System.Drawing.Imaging.ImageFormat.Png;
                        }
                        else if (rdo.Text == this.rdoTIFF.Text)
                        {
                            MIME_type = "image/tiff";
                            imageFormat = System.Drawing.Imaging.ImageFormat.Tiff;
                        }
                        else if (rdo.Text == this.rdoICO.Text)
                        {
                            // don't know the mime type for this, so can't get the EncoderParameters either :(
                            imageFormat = System.Drawing.Imaging.ImageFormat.Icon;
                        }


                        // set up the Encoder Paramaters for the save-file method
                        int encode_param_count = 0;
                        if (this.numwhlBitDepth.Visible == true) {
                            encode_param_count += 1;
                        }
                        if (this.numwhlQuality.Visible == true) {
                            encode_param_count += 1;
                        }

                        Bitmap bitmap = null;
                        string eight_bit_gif_filename = null;
                        if (encode_param_count > 0) {
                            System.Drawing.Imaging.EncoderParameters encode_params = new System.Drawing.Imaging.EncoderParameters(encode_param_count);

                            encode_param_count -= 1;    // use this as an index into the encode_params array.

                            if(this.numwhlQuality.Visible == true) {
                                // add the "Quality" percent from the number spinner.

                                encode_params.Param[encode_param_count] = new System.Drawing.Imaging.EncoderParameter(System.Drawing.Imaging.Encoder.Quality, (long)this.numwhlQuality.Value);
                                encode_param_count -= 1;
                                this.txbxDebugLog.Text += "Saved with Quality parameter " + (long)this.numwhlQuality.Value + "%\r\n";
                            }
                            if (this.numwhlBitDepth.Visible == true)
                            {
                                long bitdepth = (long)this.numwhlBitDepth.Value;

                                if((bitdepth == 8) && ((MIME_type == "image/bmp") || (MIME_type == "image/png"))) {

                                    this.txbxDebugLog.Text += "Creating 8bit gif file temporarily & deleting it\r\n";

                                    // save an 8bit gif from the open_filename.
                                    eight_bit_gif_filename = file_folder+"ImageSaverTemp8bit.gif";
                                    try
                                    {
                                        Bitmap gifsaver = new Bitmap(open_filename);
                                        gifsaver.Save(eight_bit_gif_filename, System.Drawing.Imaging.ImageFormat.Gif);    // .NET gif encoder only supports 8bit gifs
                                        gifsaver.Dispose();
                                    }
                                    catch (System.Exception exp)
                                    {
                                        this.txbxDebugLog.Text += "Temp gif conversion: Error opening file " + open_filename + "\r\n";
                                        break;
                                    }

                                    // change the   open_filename   string to point to the 8 bit gif.
                                    open_filename = eight_bit_gif_filename;

                                    // save the png as 24bit:
                                    bitdepth = 24;
                                }

                                encode_params.Param[encode_param_count] = new System.Drawing.Imaging.EncoderParameter(System.Drawing.Imaging.Encoder.ColorDepth, bitdepth);
                                encode_param_count -= 1;
                                this.txbxDebugLog.Text += "Saved with ColorDepth parameter " + bitdepth + " bits per pixel\r\n";

                            }

                            try
                            {
                                bitmap = new Bitmap(open_filename);   // the class the does the actual loading of the file, coverting it into a pure bitmap, then saving it as the specified image format!
                            }
                            catch (System.Exception excp)
                            {
                                this.txbxDebugLog.Text += "Failed to open file " + open_filename + "\r\n";
                                this.toolStripProgressBar1.Value = 0;
                                //this.Text = "ImageSaver";
                                break;
                            }

                            System.Drawing.Imaging.ImageCodecInfo codecInfo = this.GetImageCodecInfo(MIME_type);

                            bitmap.Save(save_to_filename, codecInfo, encode_params);


                        }
                        else {
                            try
                            {
                                bitmap = new Bitmap(open_filename);   // the class the does the actual loading of the file, coverting it into a pure bitmap, then saving it as the specified image format!
                            }
                            catch (System.Exception exp)
                            {
                                this.txbxDebugLog.Text += "Error opening file " + open_filename + "\r\n";
                                break;  // move onto the next file in the list
                            }

                            bitmap.Save(save_to_filename, imageFormat);
                            this.txbxDebugLog.Text += "Saved with no encoding parameters as a " + MIME_type + ":\r\n";
                        }
                        
                        if(bitmap != null)
                            bitmap.Dispose();   // done with the BitMap object.
                        if(eight_bit_gif_filename != null)
                            System.IO.File.Delete(eight_bit_gif_filename);

                        this.txbxDebugLog.Text += "\t" + save_to_filename + "\r\n";

                        this.toolStripProgressBar1.ProgressBar.Value = 75;

                        
                        // delete the original file, if the user wanted to do so:
                        if(this.chkOverWriteFiles.Checked == true) {
                            System.IO.File.Delete(filename);

                            // check if they "converted" a file into the same format
                            if (filename.Substring(period_index + 1).ToLowerInvariant() == rdo.Text.ToLowerInvariant())
                            {
                                // the original file is delete, so now rename this filenameCOPY.ext to the original file name
                                System.IO.File.Move(save_to_filename, filename);
                                save_to_filename = filename;    // the "converted" image file has a new file name now.
                                this.txbxDebugLog.Text = this.txbxDebugLog.Text.Insert(0, "Final image saved to\r\n\t" + save_to_filename + "\r\n");
                            }
                        }

                        this.toolStripProgressBar1.Value = 90;

                        // run pngcrush on the saved file.
                        if ((this.chkPNGCrush.Checked == true) && (this.chkPNGCrush.Enabled == true) && (this.chkPNGCrush.Tag != null) && (this.chkPNGCrush.Visible == true))
                        {
                            if (((string)this.chkPNGCrush.Tag).Equals("pngcrush", StringComparison.OrdinalIgnoreCase))
                            {
                                // run the file through the pngcrush program.
                                string pngcrush_path = System.Environment.CurrentDirectory + "\\pngcrush.exe";
                                if (System.Environment.GetCommandLineArgs().Length > 1)
                                {
                                    pngcrush_path = (string)(System.Environment.GetCommandLineArgs().GetValue(1));
                                }

                                // pngcrush.exe [options] infile.png outfile.png
                                string folder = System.IO.Path.GetDirectoryName(save_to_filename);
                                string file_name = System.IO.Path.GetFileNameWithoutExtension(save_to_filename);
                                string pngcrush_outfile_path = file_folder + "\\ImageSaverTempPNGcrush.png ";

                                string pngcrush_args = "-reduce " + "\"" + save_to_filename + "\" \"" + pngcrush_outfile_path + "\"";

                                System.Diagnostics.Process pngcrush = new System.Diagnostics.Process();
                                pngcrush.StartInfo.FileName = pngcrush_path;
                                pngcrush.StartInfo.Arguments = pngcrush_args;
                                //pngcrush.StartInfo.CreateNoWindow = true;
                                pngcrush.StartInfo.ErrorDialog = true;

                                try
                                {
                                    pngcrush.Start();
                                    pngcrush.WaitForExit();

                                    if (System.IO.File.Exists(pngcrush_outfile_path) == true)
                                    {
                                        System.IO.File.Delete(save_to_filename);
                                        System.IO.File.Move(pngcrush_outfile_path, save_to_filename);

                                    }
                                }
                                catch (System.Exception exp)
                                {
                                    this.txbxDebugLog.Text += exp.Message + "\r\n";
                                }
                            }
                            else if (((string)this.chkPNGCrush.Tag).Equals("optipng", StringComparison.OrdinalIgnoreCase))
                            {
                                string optipng_path = System.Environment.CurrentDirectory + "\\optipng.exe";
                                string optipng_args = "\"" + save_to_filename + "\"";

                                System.Diagnostics.Process optipng = new System.Diagnostics.Process();
                                optipng.StartInfo.FileName = optipng_path;
                                optipng.StartInfo.Arguments = optipng_args;
                                optipng.StartInfo.ErrorDialog = true;
                                try
                                {
                                    optipng.Start();
                                    optipng.WaitForExit();
                                }
                                catch (System.Exception exp)
                                {
                                    this.txbxDebugLog.Text += exp.Message + "\r\n";
                                }
                            }
                        }

                        this.toolStripProgressBar1.ProgressBar.Value = 100;


                        break;
                    }
                }
            }


            this.convertingImages = true;
            if (this.chkOverWriteFiles.Checked == true)
            {
                // remove all deleted files from the list:
                while (this.lstFiles.SelectedIndices.Count > 0)
                {
                    this.lstFiles.Items.RemoveAt(this.lstFiles.SelectedIndices[0]);
                }
            }
            this.convertingImages = false;

            this.Text = "Image Saver";
            this.toolStripProgressBar1.ProgressBar.Value = 0;
            
        }

        private void chkOverWriteFiles_CheckedChanged(object sender, EventArgs e)
        {
            if (this.chkOverWriteFiles.Checked == true)
            {
                this.lstFiles.BackColor = System.Drawing.Color.DarkRed;
                this.lstFiles.ForeColor = System.Drawing.Color.AliceBlue;
                this.btnConvertImages.BackColor = System.Drawing.Color.DarkRed;
                this.btnConvertImages.ForeColor = System.Drawing.Color.AliceBlue;

                this.tlstpOverWrite.BackColor = System.Drawing.Color.DarkRed;
                this.tlstpOverWrite.ForeColor = System.Drawing.Color.AliceBlue;

                this.btnConvertImages.Text = "Replace Images !";

                this.tlstpOverWrite.Text = "Deletes original files !!";

                this.txbxDebugLog.Text = "Selected files will be deleted after the converted image is made.\r\n\r\nThere's no confirmation, and they'll be deleted forever";
            }
            else
            {
                this.lstFiles.BackColor = System.Drawing.Color.White;
                this.lstFiles.ForeColor = System.Drawing.Color.Black;
                this.btnConvertImages.BackColor = System.Drawing.SystemColors.AppWorkspace;
                this.btnConvertImages.ForeColor = System.Drawing.SystemColors.ControlText;

                this.tlstpOverWrite.BackColor = System.Drawing.SystemColors.AppWorkspace;
                this.tlstpOverWrite.ForeColor = System.Drawing.SystemColors.ControlText;

                this.btnConvertImages.Text = "Convert Images";

                this.tlstpOverWrite.Text = "";
            }
        }

        private void trackBarOpacity_Scroll(object sender, EventArgs e)
        {
            this.Opacity = (double)this.trackBarOpacity.Value / 100.0;
            //txbxDebugLog.Text += "Opacity: " + this.Opacity + " / Scroll: " + this.trackBarOpacity.Value + "\r\n";
        }

        private void chkSelectAllFileNames_CheckedChanged(object sender, EventArgs e)
        {
            if ((bool)this.chkSelectAllFileNames.Tag == true)
            {
                this.lstFiles.BeginUpdate();
                for (int i = 0; i < this.lstFiles.Items.Count; ++i)
                {
                    this.lstFiles.SetSelected(i, true);
                }
                this.lstFiles.EndUpdate();

                this.chkSelectAllFileNames.Checked = false;
            }
            else this.chkSelectAllFileNames.Tag = true;
        }

        private void chkPictPreviewToggle_CheckedChanged(object sender, EventArgs e)
        {

            if (this.chkPictPreviewToggle.Checked == true)
            {

                // if it shrunk since we hid the pictbox
                if (this.Width < this.pictHideMainWindowWidth)
                {
                    this.Width = this.pictHideMainWindowWidth + this.pictPreviewWidth;
                }
                // if it's not wide enough to encompass the pict box now, we increase it
                else if (this.Width < (this.pictHideMainWindowWidth + this.pictPreviewWidth))
                {

                    this.Width += (this.pictHideMainWindowWidth + this.pictPreviewWidth) - this.Width;
                }

                this.pictPreview.Visible = true;

            }
            else
            {
                if (this.WindowState != FormWindowState.Maximized)
                {

                    // save the widths before hiding the window:
                    this.pictHideMainWindowWidth = this.Width - this.pictPreview.Width;
                    this.pictPreviewWidth = this.pictPreview.Width;

                    this.Width = this.Width - this.pictPreview.Width;
                    this.pictPreview.Visible = false;
                }
                else
                {
                    this.pictPreview.Visible = false;
                }
            }
        
        }

        private void chkMinWindow_CheckedChanged(object sender, EventArgs e)
        {
            if ((bool)this.chkMinWindow.Tag == true)  // prevent having this event come twice in a row
            {
                //this.txbxDebugLog.Text += "CHECK CHANGE\r\n";

                if (this.pictPreview.Visible == true)
                {
                    // save the width of main window and pict box
                    this.pictPreviewHieght = this.pictPreview.Height;
                    this.pictPreviewWidth = this.pictPreview.Width;
                    this.pictHideMainWindowWidth = this.Width - this.pictPreview.Width;

                    // turn off pict box
                    this.pictPreview.Visible = false;

                    
                    this.chkPictPreviewToggle.Checked = false;
                }

                // shrink window
                this.Height = this.MinimumSize.Height;
                this.Width = this.MinimumSize.Width;

                if ((this.chkOverWriteFiles.Location.Y - (this.lstFiles.Location.Y + this.lstFiles.Height)) > 30)
                {
                    this.lstFiles.Height += 20;
                }

                this.chkMinWindow.Tag = false;
                this.chkMinWindow.Checked = false;
            }
            else
                this.chkMinWindow.Tag = true;
        }

        private void chkHideLogTxt_CheckedChanged(object sender, EventArgs e)
        {
            if ((bool)this.chkHideLogTxt.Tag == true)   // prevent having this event come twice in a row
            {


                if (this.txbxDebugLog.ForeColor.Equals(System.Drawing.Color.DimGray) == true)
                {
                    // pop it up

                    //this.txbxDebugLog.ReadOnly = false;

                    this.txbxDebugLog.ForeColor = System.Drawing.SystemColors.WindowText;

                    this.MinimumSize = new System.Drawing.Size(870, 372);

                    // save the state of the control's location before changing.
                    this.debugLogHeight = this.txbxDebugLog.Height;
                    this.debugLogXpos = this.txbxDebugLog.Location.X;
                    this.debugLogYpos = this.txbxDebugLog.Location.Y;
                    this.resize_change_txbx.X = this.Width;
                    this.resize_change_txbx.Y = this.Height;

                    int y_offset = this.debugLogHeight - this.txbxDebugLog.MaximumSize.Height;

                    this.lstFiles.Height += y_offset;
 

                    // change the state of other controls to fill in the empty space:
                    this.chkOverWriteFiles.Location = new Point(this.chkOverWriteFiles.Location.X, this.chkOverWriteFiles.Location.Y + y_offset);
                    this.chkPictPreviewToggle.Location = new Point(this.chkPictPreviewToggle.Location.X, this.chkPictPreviewToggle.Location.Y + y_offset);


                    // change the state of it:
                    this.txbxDebugLog.Height = this.txbxDebugLog.MaximumSize.Height;
                    this.txbxDebugLog.Location = new Point(this.debugLogXpos, this.debugLogYpos + (this.debugLogHeight - this.txbxDebugLog.Height));


                    if ((this.chkOverWriteFiles.Location.Y - (this.lstFiles.Location.Y + this.lstFiles.Height)) > 30)
                    {
                        this.lstFiles.Height += 20;
                    }
                    


                }
                else
                {
                    // pull it down

                    //this.txbxDebugLog.ReadOnly = true;

                    this.MinimumSize = new System.Drawing.Size(870, 312);

                    this.txbxDebugLog.ForeColor = System.Drawing.Color.DimGray;

                    int size_offset_X = this.resize_change_txbx.X - this.Width;
                    int size_offset_Y = this.resize_change_txbx.Y - this.Height;
                    //this.debugLogXpos += size_offset_X;
                    this.debugLogYpos -= size_offset_Y;

                    
                    int y_offset = this.debugLogHeight - this.txbxDebugLog.Height;

                    this.txbxDebugLog.Location = new Point(this.debugLogXpos, this.debugLogYpos);
                    this.txbxDebugLog.Height = debugLogHeight;

                    // change the state of other controls
                    this.chkOverWriteFiles.Location = new Point(this.chkOverWriteFiles.Location.X, this.chkOverWriteFiles.Location.Y - y_offset);
                    this.chkPictPreviewToggle.Location = new Point(this.chkPictPreviewToggle.Location.X, this.chkPictPreviewToggle.Location.Y - y_offset);

                    this.lstFiles.Height -= y_offset;

                    if ((this.chkOverWriteFiles.Location.Y - (this.lstFiles.Location.Y + this.lstFiles.Height)) > 30)
                    {
                        this.lstFiles.Height += 20;
                    }
                   


                }

                this.chkHideLogTxt.Tag = false;
                this.chkHideLogTxt.Checked = false;
            }
            else
                this.chkHideLogTxt.Tag = true;

        }
    }
}